/*******************************************************************************
*
* THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTY AND SUPPORT
* IS APPLICABLE TO THIS SOFTWARE IN ANY FORM. CYTRON TECHNOLOGIES SHALL NOT,
* IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR CONSEQUENTIAL
* DAMAGES, FOR ANY REASON WHATSOEVER.
*
********************************************************************************
*
* BLD04A SAMPLE SOURCE CODE
*
*
*
* DESCRIPTION:
*
* This is the sample source code to control the BLD04A with the PIC16F877A.
* In this example, every time the SW1 is pressed and released, the motor will
* go through the following cycle:
*
* Stop naturally ---> Rotate at half speed with direction = 1 ---> Brake --->
* Rotate at full speed with direction = 0 ---> Repeat
*
* This example demonstrates the method to control the BLDC04A with parallel IO
* interface as well as the UART interface.
*
*
*
* AUTHOR   : Kong Wai Weng
* COMPANY  : Cytron Technologies Sdn Bhd
* REVISION : 1.0
* DATE     : 03/03/2010
*
*******************************************************************************/



#include <htc.h>



/*******************************************************************************
* DEVICE CONFIGURATION WORDS                                                   *
*******************************************************************************/

__CONFIG(HS &			// High Speed Crystal.
		 WDTDIS &		// Disable Watchdog Timer.
		 PWRTEN &		// Enable Power Up Timer.
		 BORDIS &		// Disable Brown Out Reset.
		 LVPDIS);		// Disable Low Voltage Programming.



/*******************************************************************************
* PRIVATE CONSTANT DEFINITION                                                  *
*******************************************************************************/

// Oscillator Frequency.
#define	_XTAL_FREQ		20000000

// UART baud rate
#define UART_BAUD		9600

// IO Connections.
#define SW1				RB0
#define START_STOP		RA1
#define RUN_BRAKE		RA2
#define DIRECTION		RA3
#define PWM				CCPR2L



/*******************************************************************************
* PRIVATE FUNCTION PROTOTYPES                                                  *
*******************************************************************************/

void uart_transmit(unsigned char uc_data);
void uart_stop_motor(void);
void uart_brake_motor(void);
void uart_run_motor(unsigned char uc_direction, unsigned int ui_speed);



/*******************************************************************************
* GLOBAL VARIABLES                                                             *
*******************************************************************************/





/*******************************************************************************
* MAIN FUNCTION                                                                *
*******************************************************************************/
int main(void)
{
	unsigned int ui_speed = 0;
	
	
	
	// Set all pins on Port A as digital IO.
	ADCON1 = 0x06;
	
	// Initialize the I/Os.
	TRISA1 = 0;				// Start/Stop output.
	TRISA2 = 0;				// Run/Brake output.
	TRISA3 = 0;				// Direction output.
	TRISB0 = 1;				// SW1 input.
	TRISC0 = 1;				// Encoder input.
	TRISC1 = 0;				// PWM output.
	TRISC6 = 0;				// UART Tx output.
	TRISC7 = 1;				// UART Tx input.
	
	PORTA = 0;
	PORTB = 0;
	PORTC = 0;
	PORTD = 0;
	PORTE = 0;
	
	
	
	// Initialize UART.
	BRGH = 1;									// Select high speed baud rate.
	SPBRG = (_XTAL_FREQ / 16 / UART_BAUD) - 1;	// Configure the baud rate.
	SPEN = 1;									// Enable serial port.
	CREN = 1;									// Enable reception.
	TXEN = 1;									// Enable transmission.
	
	
	
	// Initialize PWM.
	PR2 = 0xff;
	T2CKPS1 = 0;			// Setting PWM frequency = 4.88khz.
	T2CKPS0 = 1;			// Timer 2 prescale = 4.
	PWM = 0;				// Duty cycle = 0;
	TMR2ON = 1;				// Turn on Timer 2.
	CCP2M3 = 1;
	CCP2M2 = 1;				// Configure CCP2 module to operate in PWM mode.
	
	
	
	// Infinite loop.
	while (1)
	{
		// Stop the motor naturally (Free-Running Stop).
		// Using parallel IO interface.
		START_STOP = 1;
		RUN_BRAKE = 0;
		
		// Using UART.
		uart_stop_motor();
		
		
		
		// Wait for user to press and then release SW1.
		while (SW1 == 1);
		while (SW1 == 0);
		
		
		
		// Run the motor at half speed with direction = 1.
		// Using parallel IO interface.
		DIRECTION = 1;
		PWM = 128;
		START_STOP = 0;
		RUN_BRAKE = 0;
		
		// Using UART.
		uart_run_motor(1, 1500);
		
		
		
		// Wait for user to press and then release SW1.
		while (SW1 == 1);
		while (SW1 == 0);
		
		
		
		// Stop the motor instantaneously (Brake).
		// Using parallel IO interface.
		START_STOP = 1;
		RUN_BRAKE = 1;
		
		// Using UART.
		uart_brake_motor();
		
		
		
		// Wait for user to press and then release SW1.
		while (SW1 == 1);
		while (SW1 == 0);
		
		
		
		// Run the motor at full speed with direction = 0.
		// Using parallel IO interface.
		DIRECTION = 0;
		PWM = 255;
		START_STOP = 0;
		RUN_BRAKE = 0;
		
		// Using UART.
		uart_run_motor(0, 3000);
		
		
		
		// Wait for user to press and then release SW1.
		while (SW1 == 1);
		while (SW1 == 0);
	}	
}



/*******************************************************************************
* PRIVATE FUNCTION: uart_transmit
*
* PARAMETERS:
* ~ uc_data		- The data that we want to transmit.
*
* RETURN:
* ~ void
*
* DESCRIPTIONS:
* This function will transmit one byte of data using UART. This is a blocking
* function, if the transmit buffer is full, we will wait until the
* data in the buffer has been sent out before we move in the new data.
*
*******************************************************************************/
void uart_transmit(unsigned char uc_data)
{
	// Wait until the transmit buffer is ready for new data.
	while (TXIF == 0);
	
	// Transmit the data.
	TXREG = uc_data;
}



/*******************************************************************************
* PRIVATE FUNCTION: uart_stop_motor
*
* PARAMETERS:
* ~ void
*
* RETURN:
* ~ void
*
* DESCRIPTIONS:
* This function will transmit the command to the BLD04A via UART to stop the
* motor naturally (Free-Running Stop).
*
*******************************************************************************/
void uart_stop_motor(void)
{
	// Send 2 bytes of header.
	uart_transmit(0x00);
	uart_transmit(0x00);
	
	// Send the command to stop the motor.
	uart_transmit(0xf0);
}



/*******************************************************************************
* PRIVATE FUNCTION: uart_brake_motor
*
* PARAMETERS:
* ~ void
*
* RETURN:
* ~ void
*
* DESCRIPTIONS:
* This function will transmit the command to the BLD04A via UART to stop the
* motor instantaneously (Brake).
*
*******************************************************************************/
void uart_brake_motor(void)
{
	// Send 2 bytes of header.
	uart_transmit(0x00);
	uart_transmit(0x00);
	
	// Send the command to brake the motor.
	uart_transmit(0xf1);
}



/*******************************************************************************
* PRIVATE FUNCTION: uart_run_motor
*
* PARAMETERS:
* ~ uc_direction	- The direction of the motor (0 or 1).
* ~ ui_speed		- The speed of the motor (0 to 3000).
*
* RETURN:
* ~ void
*
* DESCRIPTIONS:
* This function will transmit the command to the BLD04A via UART to set the
* direction of the motor, set the speed of the motor and then run the motor.
*
*******************************************************************************/
void uart_run_motor(unsigned char uc_direction, unsigned int ui_speed)
{
	// Send 2 bytes of header.
	uart_transmit(0x00);
	uart_transmit(0x00);
	
	// Send the command to set the direction accordingly.
	if (uc_direction == 0)
	{
		uart_transmit(0xf3);
	}
	else
	{
		uart_transmit(0xf4);
	}	
	
	
	
	// Send 2 bytes of header.
	uart_transmit(0x00);
	uart_transmit(0x00);	
	
	// Send the command to set the speed.
	uart_transmit(0xf6);
	uart_transmit(ui_speed >> 8);
	uart_transmit(ui_speed & 0xff);
	
	
	
	// Send 2 bytes of header.
	uart_transmit(0x00);
	uart_transmit(0x00);
		
	// Send the command to run the motor.
	uart_transmit(0xf2);
}
